########################################################################
#
#    zgdchart.py
#
#    (c) Cognoware Australia P/L
#
#    Author: Chui Tey <teyc@cognoware.com>
#
#    Acknowledgements and sincere thanks!:
#    
#    Hung Jung Lu <hungjunglu@hotmail.com> (version 0.4.2.b) - new data method, RPM package
#    Andreas Furbach <andreas.furbach@eds.com> (version 0.5) - cleaner data implementation
#    Marcus Bursik <mib@geology.buffalo.edu>           - gdchart binaries for solaris
#    Eric Maryniak <e.maryniak@pobox.com>            - xhtml compliance in demo
#
#    To do:
#        gdchartdemo.index_html is not displaying all the graphs in one go on my slow machine
#        support multiple columns on SQL queries
#        a color wheel on the editColor properties
#        dtml method creates a frame which is not removed later
#
# ======================================================================
#
#  Disclaimer
#
#   Redistribution and use in source and binary forms, with or without
#   modification, are permitted provided that the following conditions
#   are met:
#
#     Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#
#     Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in the
#     documentation and/or other materials provided with the distribution.
#
#     The name of the author may not be used to endorse or promote 
#     products derived from this software without specific prior
#     written permission
#
#   THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY
#   EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
#   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
#   PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR THEIR
#   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
#   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
#   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
#   USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
#   ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
#   OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
#   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
#   SUCH DAMAGE.
#
# ======================================================================
__version__='0.6.4'

import Acquisition
import AccessControl
import OFS
import urllib
import string
import cStringIO
import os
from Globals import HTMLFile, MessageDialog, Persistent
from Globals import package_home
from ImageFile import ImageFile
from DocumentTemplate.DT_Util import *
from DocumentTemplate.DT_String import String
from   zLOG import LOG, WARNING, INFO, ERROR, DEBUG

#
# In this module, extend gdchart so
# that it can render itself
#
from gdchart import *
import GDChart

manage_addZGDChartForm=HTMLFile('manage/addZGDChart',globals())

def manage_addZGDChart(self, id, title, REQUEST=None):
    """
    Create a ZGDChart and install it in its parent Folder
    The argument 'self' will be bound to the parent Folder.
    """

    handle=ZGDChart(id, title)
    handle.id=id
    handle.title=title
    self._setObject(id,handle)
    if REQUEST is not None:
        return self.manage_main(self, REQUEST)

def parse_version(version):
    """Given a version of the format 'ZGDChart 0.4.2', parse
    into a list [0.0 , 4.0, 2.0]"""
    version_numbers = string.split(version,' ')[1]
    version_numbers = string.split(version_numbers, '.')
    l=[]
    for item in version_numbers:
        try:
            l.append(float(item))
        except:
            pass
        return l

class ZGDChart(
    GDChart.GDChart,
    Acquisition.Implicit,
    Persistent,
    AccessControl.Role.RoleManager,
    OFS.SimpleItem.Item,
    OFS.ObjectManager.ObjectManager
    ):
    '''
    A simple chart class which plots data from SQL, TinyTable, DTML, or
    PythonScript sources.

    The chart plots the first two columns of a SQL query result.

    This product uses gdchart0.94b, which in turn uses gd1.3
    The current version of gd 1.7.3 does not support gif creation due to the Unisys patent.

    Wrapper authored by <a href="mailto:teyc@cognoware.com">Chui Tey</a>.
    
    (c) Cognoware Australia P/L

    Software is provided without warranty.

    Credits:

    gdchart      - Bruce Verdeaime   <A HREF="http://www.fred.net/brv/chart/">http://www.fred.net/brv/chart/</A>
    
    gdchart.py   - Michael Ray Steed <A HREF="http://members.xoom.com/_XMCM/msteed/software/gdchart.html"> http://members.xoom.com/_XMCM/msteed/software/gdchart.html </A>

    zgdchart.py  - Hung Jung Lu <hungjunglu@hotmail.com> (version 0.4.2.b) - new data method, RPM package
    
    This product was created using mkproduct 0.6 and Zope Simple Skeleton v0.2,
    written by Sebastian L&uuml;hnsdorf (<a href="mailto:basti@beehive.de">basti@beehive.de</a>)

    For help, refer to the <A href="http://localhost/HelpSys/ObjectRef/ZGDChart/hs_main">Zope help system</A>
    '''

    name      =    'ZGDChart'
    id        =    'ZGDChart'
    icon      =    'misc_/ZGDChart/Main'
    meta_type =    'ZGDChart'

    _properties = (
        {'id':'title','type':'string','mode':'w'},
        )

    manage_options=(
        {'label':'Properties' , 'icon':icon, 'action':'manage_main',        'target':'_self', 'help': ('ZGDChart','ZGDChart_Properties.stx')},
        {'label':'Colors'     , 'icon':icon, 'action':'manage_mainColor',   'target':'_self', 'help': ('ZGDChart','ZGDChart_Colors.stx')},
        {'label':'Scale',       'icon':icon, 'action':'manage_mainScale',   'target':'_self', 'help': ('ZGDChart','ZGDChart_Scale.stx')},
        {'label':'Data Method', 'icon':icon, 'action':'manage_mainGetData', 'target':'_self', 'help': ('ZGDChart','ZGDChart_DataMethod.stx')},
        {'label':'Pie Option' , 'icon':icon, 'action':'manage_mainPie',     'target':'_self', 'help': ('ZGDChart','ZGDChart_DataMethod.stx')},
        {'label':'Security'   , 'icon':icon, 'action':'manage_access',      'target':'_self', 'help': ('ZGDChart','ZGDChart_Security.stx')},
        {'label':'View'       , 'icon':icon, 'action':'manage_mainView',    'target':'_self'},
        )

    __ac_permissions__=(
        ('View management screens',('manage_tabs','manage_main')),
        ('Change permissions',('manage_access',)),
        ('Change ZGDChart',('manage_edit','manage_editColor','manage_pieOption'),('Manager',)),
        )

    # External HTML files
    
    manage_main = HTMLFile('manage/edit', globals())
    manage_mainColor = HTMLFile('manage/editColor', globals())
    manage_mainScale = HTMLFile('manage/editScale', globals())
    manage_mainGetData = HTMLFile('manage/getData', globals())
    manage_mainView = HTMLFile('manage/view', globals())
    manage_mainPie = HTMLFile('manage/editPie',globals())
    upper_manage_mainGetData = HTMLFile('manage/getData', globals())
    lower_manage_mainSelectZsql = HTMLFile('manage/getZsqlData', globals())

    # REQUEST runtime parameter prefix
    runtime_prefix = 'zgdchart_runtime_'

    def home(self):
        "returns the directory where the product is located in"
        return package_home(globals())
    
    def version (self):
        "Return the version number of this product"        
        return open(self.home() + '/version.txt').read()
    
    def version_tuple(self):
        return parse_version(self.version())

    def __init__(self, id='ZGDChart', title='ZGDChart', height=250, width=250, 
        SQL=None, chart_type='Bar_3D', option=['xaxis', 'yaxis', 'grid', 'border']):
        
        GDChart.GDChart.__init__(self, id, title, height, width, SQL, chart_type, option)

        # CGT: 10/09/03 Probably no longer required
        #self.tempfile       = self.home() + '/temp.gif'

        #store the version number of the current instance
        self.version_tuple_original   = self.version_tuple()

        self.data_source = ""

        # Set the data gathering method
        # to be deprecated
        self.selected_data_method = 'dtml'
        f = open(os.path.join(self.home(),'manage', 'getDtmlData.dtml'), 'r')
        getDtmlData_dtml = f.read()
        f.close()
        ob = OFS.DTMLMethod.DTMLMethod(getDtmlData_dtml, __name__='getDtmlData')
        self._setObject('getDtmlData', ob)
            
    def _upgrade(self):
        """ Technique to bring old instances up to date with any new required attributes """
        if not hasattr(self, 'selected_data_method'):
            self.selected_data_method = 'zsql'
        if not hasattr(self, 'set_color_list'):
            self.set_color_list = self.pie_color
        if not hasattr(self, 'set_color'):
            self.set_color = self.pie_color
        if not hasattr(self, 'script'):
            self.script = None
        if not hasattr(self, 'TinyTable'):
            self.TinyTable = None
        if len(getattr(self, 'explode', [])) != 6:
            self.explode = [0,0,0,0,0,15]
        self.version_tuple_original = self.version_tuple()

    def __setstate__(self, state):               
        Persistent.__setstate__(self, state) # Important, always call this
        # Check if we need to upgrade the current instance
        #
        if hasattr(self, 'version_tuple_original'):
            if self.version_tuple_original < self.version_tuple():
                self._upgrade()
        else:
            self.upgrade() # heh, must be a very old version
            
    def manage_edit(self, title, xtitle, ytitle, width, height, chart_type, option=[], REQUEST=None):
        """edit ZGDChart"""
        self.title          = title
        self.xtitle         = xtitle
        self.ytitle         = ytitle
        self.height         = height
        self.width          = width
        self.chart_type     = chart_type
        self.bg_transparent = 'bg_transparent' in option
        self.label_line     = 'label_line' in option
        self.thumbnail      = 'thumbnail' in option
        self.border         = 'border' in option
        self.grid           = 'grid' in option
        self.xaxis          = 'xaxis' in option
        self.yaxis          = 'yaxis' in option
        self.yaxis2         = 'yaxis2' in option

        if REQUEST is not None:
            return MessageDialog(
                title='Edited',
                message='<strong>%s</strong> has been edited.' % self.id,
                action = './manage_main'
                )

    def manage_editColor(self, bg_color, edge_color, grid_color, plot_color,
                         title_color, xtitle_color, ytitle_color, ytitle2_color,
                         xlabel_color, ylabel_color, ylabel2_color, set_color_list,
                         pie_color_list,
                         REQUEST=None):
        """ edit ZGDChart colors
            all values to be passed as strings in hex, eg..
            manage_editColor(bg_color = "0xFFFFFF")
        """
        self.bg_color       = string.atoi(bg_color,   16)
        self.edge_color     = string.atoi(edge_color, 16)
        self.grid_color     = string.atoi(grid_color, 16)
        self.plot_color     = string.atoi(plot_color, 16)
        self.title_color    = string.atoi(title_color, 16)
        self.xtitle_color   = string.atoi(xtitle_color, 16)
        self.ytitle_color   = string.atoi(ytitle_color, 16)
        self.ytitle2_color  = string.atoi(ytitle2_color, 16)
        self.xlabel_color   = string.atoi(xlabel_color, 16)
        self.ylabel_color   = string.atoi(ylabel_color, 16)
        self.ylabel2_color  = string.atoi(ylabel2_color, 16)
        self.set_color      = map(lambda x:string.atoi(x,16), string.split(set_color_list, ','))
        self.pie_color      = map(lambda x:string.atoi(x,16), string.split(pie_color_list, ','))
        
        if REQUEST is not None:
            return MessageDialog(
                title='Edited',
                message='<strong>%s</strong> has been edited.' % self.id,
                action = './manage_mainColor'
                )

    def manage_pieOption(self, threed_angle=0, 
        threed_depth=0, percent_labels=GDCPIE_PCT_ABOVE, 
        label_dist=0, list_explode='0,0,0,0', REQUEST=None):
        """ edit ZGDChart option pie
        """
        self.threed_angle=threed_angle
        self.threed_depth=threed_depth
        tab_percent=[GDCPIE_PCT_NONE,GDCPIE_PCT_ABOVE,GDCPIE_PCT_BELOW,GDCPIE_PCT_RIGHT,GDCPIE_PCT_LEFT]
        self.percent_labels=tab_percent[int(percent_labels)]
        self.label_dist=label_dist
        self.explode=string.split(list_explode, ',')
        self.explode=[int(x) for x in  self.explode]
        if REQUEST is not None:
            return MessageDialog(
                title='Edited',
                message='<strong>%s</strong> has been edited.' % self.id,
                action = './manage_mainPie'
                )
        

    def manage_editScale(self,
                 requested_ymax=None, requested_ymin=None, requested_yinterval=None,
                 xlabel_spacing=5, ylabel_density=80,
                 ylabel_fmt='', ylabel2_fmt='', REQUEST=None):
        """ edit ZGDChart scale properties """        
        self.xlabel_spacing         =    xlabel_spacing
        self.requested_ymax         =    requested_ymax        
        self.requested_ymin         =    requested_ymin
        self.requested_yinterval    =    requested_yinterval
        self.ylabel_density         =    ylabel_density
        self.ylabel_fmt             =    ylabel_fmt
        self.ylabel2_fmt            =    ylabel2_fmt

        if REQUEST is not None:
            return MessageDialog(
                title='Edited',
                message='<strong>%s</strong> has been edited.' % self.id,
                action = './manage_mainScale'
                )

    def manage_changeDataSource(self, data_source):
        """ Change the data source """
        self.data_source = data_source
        self.REQUEST.RESPONSE.redirect('manage_mainGetData')

    def manage_changeDataMethod(self, data_method, REQUEST=None):
        """change Data Method"""
        self.selected_data_method = data_method
        if REQUEST is not None:
            return MessageDialog(
                title='Data Method Changed',
                message='Data Method for <strong>%s</strong> has been changed.' % self.id,
                action = './manage_mainGetData',
                target = 'manage_main'
                )

    def manage_selectSQL(self, SQL, REQUEST=None):
        """select SQL"""
        self.SQL = SQL
        if REQUEST is not None:
            return MessageDialog(
                title='Selected',
                message='Selection has been recorded.',
                action = './manage_mainGetData',
                target = 'manage_main'
                )

    def manage_selectScript(self, script, REQUEST=None):
        "select Script"""
        self.script = script
        if REQUEST is not None:
            return MessageDialog(
                title='Selected',
                message='Selection has been recorded.',
                action='./manage_mainGetData',
                target='manage_main')

    def manage_selectTinyTable(self, TinyTable, Columns=[], Filter={}, REQUEST=None):
        """select TinyTable as data source"""
        # Columns and Filter are unused at the moment
        self.TinyTable = TinyTable
        self.TinyTable_columns = Columns
        self.TinyTable_filter = Filter
        if REQUEST is not None:
            return MessageDialog(
                title='Selected',
                message='Selection has been recorded.',
                action = './manage_mainGetData',
                target = 'manage_main'
                )

    def __call__(self, mapping={}, **kwargs):
        ''' Just another version of __str__ '''
        # Not sure if this is going to block other inherited methods

        # combine mapping and kwargs into a single dictionary
        dict = {}
        for key,value in mapping.items():
            dict[key]=value
        for key,value in kwargs.items():
            dict[key]=value
        
        # roll our own urlencode because we want
        # / to be encoded
        # query = urllib.urlencode(kwargs)
        query = self.urlencode(dict)

        #
        # get the actual path to the object in case we are
        # not calling from the same directory
        #
        path = string.join(self.getPhysicalPath(), '/')
        return '<IMG SRC="%s?%s" ALT="%s">' % (path, query, self.title)

    def urlencode(self, dict):
        #
        # have to quote lists and tuples in format Zope understands
        # therefore test for types in dictionary
        #
        from types import ListType, TupleType
        
        l = []
        #import pdb; pdb.set_trace()
        for k, v in dict.items():
            
            if type(v) is ListType or type(v) is TupleType:
                k = urllib.quote_plus(str(k+':tokens'))
                v_str = ""
                for item in v:
                    v_str = v_str + "+" + urllib.quote_plus(str(item))
                l.append(k + '=' + v_str)
            else:
                k = urllib.quote_plus(str(k))
                v = urllib.quote_plus(str(v),'')
                l.append(k + '=' + v)

        return string.join(l, '&')

    # following routines are inspired by Arpad Kiss's chart product
    def __str__(self,REQUEST=None,RESPONSE=None):
        """
        Returns a string representation of the object <IMG SRC=xxx>
        """
        return '<IMG SRC="%s" ALT="%s">' % (self.id, self.title)
    
    def index_html(self,REQUEST=None,RESPONSE=None):
        """
        This is the default method.
        """
        #
        # The default chart type can be overridden by passing
        # zgdchart_runtime_type = 'Bar_2D' etc.
        #
        chart_type = self.getRunTimeOption('type')
        return self._chart(ZGDChart.chart_type_codes[chart_type], RESPONSE)        
        #
        # return the default chart
        #
        return self._chart(ZGDChart.chart_type_codes[chart_type], RESPONSE)

    def _chartdata(self):
        if getattr(self, 'data_source', None):
            data_source = self.data_source
            try:
                # DTML Method
                data = getattr(self, data_source)(self.REQUEST)
            except:
                data = getattr(self, data_source)()
            chartdata = self._formatData(data)                
        elif self.selected_data_method == 'dtml':
            chartdata = self.getDtmlData(self, self.REQUEST)
        elif self.selected_data_method == 'pyscript':
            chartdata = self._chartdata_script()
        elif self.selected_data_method == 'tinytable':
            chartdata = self._chartdata_TinyTable()
        elif self.selected_data_method == 'zsql':
            chartdata = self._chartdata_SQL()
        return chartdata            

    def _chartdata_DTML(self):
        DTML = getattr(self, self.DATASOURCE_ID)

    def _chartdata_SQL(self):
        SQL = getattr(self, self.SQL)
        try: rows = SQL()
        except:
            LOG("ZGDChart", INFO, SQL(src__=1))
            raise
        return self._formatData(rows)

    def _chartdata_script(self):
        script = getattr(self, self.script)
        rows = script()
        return self._formatData(rows)
        
    def _chartdata_TinyTable(self):
        #
        # gets chart data from Tiny Table
        # uses self.TinyTable
        #      self.TinyTable_columns - list
        #      self.TinyTable_filter  - dict
        #
        TinyTable = getattr(self, self.TinyTable)
        rows = apply(TinyTable, self.TinyTable_filter)
        return self._formatData(rows)

    # XXX not a good idea? hit SQL servers too many times
    def _labels(self):
        """
        return the first column sql query results as a tuple
        """
        rows = getattr(self, self.SQL)()
        tmp=()
        for row in rows:
           tmp = tmp + (row[0],)
        return tmp
        
    def _data(self):
        rows = getattr(self, self.SQL)()
        tmp=()
        for row in rows:
            tmp = tmp + (row[1],)
        return tmp

    def initRunTimeOption(self):
        # Overrides GDChart.initRunTimeOption
        pass

    def getRunTimeOption(self, key):
        # Overrides GDChart.getRunTimeOption

        if key == 'type':
            return self.REQUEST.get(self.runtime_prefix + key, \
                    getattr(self, 'chart_type'))
            
        return self.REQUEST.get(self.runtime_prefix + key, \
                getattr(self, key))

    def setRunTimeOption(self, key, value):
        self.REQUEST[self.runtime_prefix + 'key'] = value

# vim: set et sta ai ts=4 sw=4 sts=4

